<?php
// functions.php
require_once __DIR__ . '/config.php';

// ==== BASIC HELPER ====
function h($str) {
    return htmlspecialchars($str, ENT_QUOTES, 'UTF-8');
}

// ==== AUTH ====
function get_current_user_row() {
    global $mysqli;
    if (empty($_SESSION['uid'])) return null;

    static $cache = null;
    if ($cache !== null) return $cache;

    $id = (int)$_SESSION['uid'];
    $stmt = $mysqli->prepare("SELECT * FROM " . TBL_USERS . " WHERE id=? LIMIT 1");
    $stmt->bind_param('i', $id);
    $stmt->execute();
    $res = $stmt->get_result();
    $cache = $res->fetch_assoc() ?: null;
    $stmt->close();
    return $cache;
}

function require_login() {
    if (!get_current_user_row()) {
        header('Location: admin.php');
        exit;
    }
}

function is_super_admin() {
    $u = get_current_user_row();
    return $u && (int)$u['is_super'] === 1;
}

// ==== SETTINGS (GLOBAL & PER ROTATOR) ====
function get_global_setting($key, $default = '') {
    global $mysqli;
    $stmt = $mysqli->prepare("SELECT value_text FROM " . TBL_SETTINGS_GLOBAL . " WHERE key_name=? LIMIT 1");
    $stmt->bind_param('s', $key);
    $stmt->execute();
    $res = $stmt->get_result();
    $row = $res->fetch_assoc();
    $stmt->close();
    return $row ? $row['value_text'] : $default;
}

function set_global_setting($key, $value) {
    global $mysqli;
    $stmt = $mysqli->prepare("
        INSERT INTO " . TBL_SETTINGS_GLOBAL . " (key_name,value_text)
        VALUES(?,?)
        ON DUPLICATE KEY UPDATE value_text=VALUES(value_text)
    ");
    $stmt->bind_param('ss', $key, $value);
    $stmt->execute();
    $stmt->close();
}

function get_rotator_setting($rotator_id, $key, $default = '') {
    global $mysqli;
    $stmt = $mysqli->prepare("SELECT value_text FROM " . TBL_SETTINGS_ROTATOR . " WHERE rotator_id=? AND key_name=? LIMIT 1");
    $stmt->bind_param('is', $rotator_id, $key);
    $stmt->execute();
    $res = $stmt->get_result();
    $row = $res->fetch_assoc();
    $stmt->close();
    return $row ? $row['value_text'] : $default;
}

function set_rotator_setting($rotator_id, $key, $value) {
    global $mysqli;
    $stmt = $mysqli->prepare("
        INSERT INTO " . TBL_SETTINGS_ROTATOR . " (rotator_id,key_name,value_text)
        VALUES(?,?,?)
        ON DUPLICATE KEY UPDATE value_text=VALUES(value_text)
    ");
    $stmt->bind_param('iss', $rotator_id, $key, $value);
    $stmt->execute();
    $stmt->close();
}

// ==== TELEGRAM ====
function send_telegram_message($text, $chat_id) {
    if (!TG_BOT_TOKEN || !$chat_id) return;

    $url = "https://api.telegram.org/bot" . TG_BOT_TOKEN . "/sendMessage";
    $data = [
        'chat_id' => $chat_id,
        'parse_mode' => 'HTML',
        'text' => $text,
        'disable_web_page_preview' => true,
    ];

    @file_get_contents($url . '?' . http_build_query($data));
}

function notify_rotator_owner($rotator_row, $text) {
    global $mysqli;
    $uid = (int)$rotator_row['user_id'];
    $stmt = $mysqli->prepare("SELECT telegram_chat_id, notify_enabled FROM " . TBL_USERS . " WHERE id=? LIMIT 1");
    $stmt->bind_param('i', $uid);
    $stmt->execute();
    $res = $stmt->get_result();
    $u = $res->fetch_assoc();
    $stmt->close();
    if (!$u) return;

    if (!empty($u['notify_enabled']) && !empty($u['telegram_chat_id'])) {
        send_telegram_message($text, $u['telegram_chat_id']);
    }
}

// ==== NAWALA CHECK ====
// kamu ganti logika sesuai API kamu (GET/POST, dll)
function check_trustpositif($domain) {
    $domain = preg_replace('#^https?://#i', '', trim($domain));
    if (!$domain) {
        return [
            'blocked' => null,
            'message' => 'Domain kosong',
        ];
    }

    if (!NWL_API_URL) {
        // fallback: anggap aman (tidak terblokir)
        return [
            'blocked' => false,
            'message' => 'CHECK DISABLED (NWL_API_URL kosong)',
        ];
    }

    $params = [
        'domain' => $domain,
        'secret' => NWL_API_SECRET,
    ];
    $url = NWL_API_URL . '?' . http_build_query($params);

    $resp = @file_get_contents($url);
    if ($resp === false) {
        return [
            'blocked' => null,
            'message' => 'Gagal konek API Nawala',
        ];
    }

    // Sesuaikan dengan format JSON API kamu
    $data = @json_decode($resp, true);
    if (!$data) {
        return [
            'blocked' => null,
            'message' => 'Response API tidak valid',
        ];
    }

    // CONTOH: { "blocked": true/false, "info": "..." }
    $blocked = isset($data['blocked']) ? (bool)$data['blocked'] : null;
    $msg = isset($data['info']) ? $data['info'] : 'OK';

    return [
        'blocked' => $blocked,
        'message' => $msg,
    ];
}
